<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

requireLogin();
$activePage = 'dashboard';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Check if user has permission to view dashboard
$isSuperAdmin = false;
if (isset($_SESSION['user_id'])) {
    $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = (SELECT role_id FROM users WHERE id = ?)");
    $stmt->execute([$_SESSION['user_id']]);
    $isSuperAdmin = $stmt->fetchColumn();
}

$hasDashboardPermission = hasPermission('view_dashboard') || $isSuperAdmin;
$userName = $_SESSION['username'] ?? 'User';

// Get date range for analytics
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

// After $end_date is set
$today = date('Y-m-d');

// Get store settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
$stmt->execute();
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get sales statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_sales,
        SUM(total_amount) as total_revenue,
        SUM(CASE WHEN payment_method = 'credit' THEN total_amount ELSE 0 END) as credit_sales,
        SUM(CASE WHEN payment_method = 'cash' THEN total_amount ELSE 0 END) as cash_sales,
        COUNT(DISTINCT customer_id) as total_customers
    FROM sales 
    WHERE created_at BETWEEN ? AND ? AND status = 'completed'
");
$stmt->execute([$start_date . ' 00:00:00', $end_date . ' 23:59:59']);
$stats = $stmt->fetch();

// Get daily sales for chart
$stmt = $pdo->prepare("
    SELECT 
        DATE(created_at) as date,
        SUM(total_amount) as total,
        SUM(CASE WHEN payment_method = 'credit' THEN total_amount ELSE 0 END) as credit_total
    FROM sales 
    WHERE created_at BETWEEN ? AND ? AND status IN ('completed', 'pending')
    GROUP BY DATE(created_at)
    ORDER BY date
");
$stmt->execute([$start_date . ' 00:00:00', $end_date . ' 23:59:59']);
$daily_sales = $stmt->fetchAll();

// Get today's credit sales: total balance of credit sales created today (resets daily at midnight)
$stmt = $pdo->prepare("
    WITH todays_credit_balances AS (
        SELECT 
            ct.id,
            ct.amount,
            COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
            ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
        FROM credit_transactions ct
        JOIN sales s ON ct.sale_id = s.id
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        WHERE DATE(s.created_at) = ?
        GROUP BY ct.id, ct.amount
    )
    SELECT COALESCE(SUM(balance), 0) as todays_credit_balance
    FROM todays_credit_balances
");
$stmt->execute([$today]);
$today_credit_sales = $stmt->fetchColumn();

// Get today's profit
$stmt = $pdo->prepare("
    SELECT 
        COALESCE(SUM(s.total_amount), 0) as total_revenue,
        COALESCE(SUM(si.quantity * p.cost_price), 0) as total_cost
    FROM sales s
    JOIN sale_items si ON s.id = si.sale_id
    JOIN products p ON si.product_id = p.id
    WHERE DATE(s.created_at) = ? 
    AND s.status = 'completed'
");
$stmt->execute([$today]);
$profit_data = $stmt->fetch();
$today_profit = $profit_data['total_revenue'] - $profit_data['total_cost'];

// Get top selling products
$stmt = $pdo->prepare("
    SELECT 
        p.name,
        SUM(si.quantity) as total_quantity,
        SUM(si.quantity * si.unit_price) as total_revenue
    FROM sale_items si
    JOIN products p ON si.product_id = p.id
    JOIN sales s ON si.sale_id = s.id
    WHERE s.created_at BETWEEN ? AND ? AND s.status = 'completed'
    GROUP BY p.id
    ORDER BY total_quantity DESC
    LIMIT 5
");
$stmt->execute([$start_date . ' 00:00:00', $end_date . ' 23:59:59']);
$top_products = $stmt->fetchAll();

// Get worst selling products
$stmt = $pdo->prepare("
    SELECT 
        p.name,
        SUM(si.quantity) as total_quantity,
        SUM(si.quantity * si.unit_price) as total_revenue
    FROM sale_items si
    JOIN products p ON si.product_id = p.id
    JOIN sales s ON si.sale_id = s.id
    WHERE s.created_at BETWEEN ? AND ? AND s.status = 'completed'
    GROUP BY p.id
    ORDER BY total_quantity ASC
    LIMIT 5
");
$stmt->execute([$start_date . ' 00:00:00', $end_date . ' 23:59:59']);
$worst_products = $stmt->fetchAll();

// Get recent transactions
$stmt = $pdo->prepare("
    SELECT 
        s.*,
        c.name as customer_name,
        u.username as cashier_name
    FROM sales s
    LEFT JOIN customers c ON s.customer_id = c.id
    JOIN users u ON s.user_id = u.id
    WHERE s.created_at BETWEEN ? AND ?
    ORDER BY s.created_at DESC
    LIMIT 10
");
$stmt->execute([$start_date . ' 00:00:00', $end_date . ' 23:59:59']);
$recent_transactions = $stmt->fetchAll();

// Get low stock products
$stmt = $pdo->prepare("
    SELECT 
        name,
        quantity,
        alert_quantity
    FROM products
    WHERE quantity <= alert_quantity
    ORDER BY quantity ASC
    LIMIT 10
");
$stmt->execute();
$low_stock = $stmt->fetchAll();

// Get pending credit sales
$stmt = $pdo->prepare("
    SELECT 
        s.*,
        c.name as customer_name,
        ct.amount as credit_amount,
        COALESCE(SUM(cp.amount), 0) as amount_paid
    FROM sales s
    JOIN customers c ON s.customer_id = c.id
    JOIN credit_transactions ct ON s.id = ct.sale_id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    WHERE s.payment_method = 'credit' AND s.payment_status = 'pending' AND s.created_at BETWEEN ? AND ?
    GROUP BY s.id
    ORDER BY s.created_at DESC
    LIMIT 5
");
$stmt->execute([$start_date . ' 00:00:00', $end_date . ' 23:59:59']);
$pending_credits = $stmt->fetchAll();

// Get today's total cash sales AMOUNT
$today = date('Y-m-d');
$stmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount),0) as total_sales_amount FROM sales WHERE DATE(created_at) = ? AND payment_method = 'cash' AND status = 'completed'");
$stmt->execute([$today]);
$today_cash_sales_amount = $stmt->fetchColumn();

// Get today's total credit payments (actual payments made with method 'credit')
$stmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM sale_payments WHERE method = 'credit' AND DATE(created_at) = ?");
$stmt->execute([$today]);
$today_credit_payments = $stmt->fetchColumn();

// Get total number of customers
$stmt = $pdo->prepare("SELECT COUNT(DISTINCT customer_id) FROM sales WHERE created_at BETWEEN ? AND ? AND status = 'completed'");
$stmt->execute([$start_date . ' 00:00:00', $end_date . ' 23:59:59']);
$total_customers = $stmt->fetchColumn();

// Get favicon path from database
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'store_favicon' AND setting_group = 'store_info' LIMIT 1");
$stmt->execute();
$faviconPath = $stmt->fetchColumn();

// Add after the other stat queries
$stmt = $pdo->prepare("SELECT COUNT(*) FROM suppliers");
$stmt->execute();
$total_suppliers = $stmt->fetchColumn();

// Get total credit amount using same calculation as credit.php (unpaid credit items)
$stmt = $pdo->prepare("
    WITH credit_balances AS (
        SELECT 
            ct.customer_id,
            ct.amount,
            COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
            ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
        FROM credit_transactions ct
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        GROUP BY ct.id, ct.customer_id, ct.amount
        HAVING balance > 0
    )
    SELECT COALESCE(SUM(balance), 0) as total_pending
    FROM credit_balances
");
$stmt->execute();
$total_credit_amount = $stmt->fetchColumn();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Dashboard - EDDUH COMPUTERS</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="../assets/css/responsive.css" rel="stylesheet">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body {
            background: linear-gradient(135deg, #e0e7ff 0%, #f8fafc 100%);
            font-family: 'Inter', sans-serif;
            min-height: 100vh;
        }
        .modern-stat-card {
            background: rgba(255,255,255,0.85);
            backdrop-filter: blur(8px);
            border-radius: 16px;
            box-shadow: 0 8px 32px rgba(37,99,235,0.13), 0 1.5px 6px rgba(0,0,0,0.03);
            border-left: 4px solid #3874ff;
            padding: 1rem 0.8rem 0.8rem 0.8rem;
            margin-bottom: 1.5rem;
            min-height: 100px;
            position: relative;
            overflow: hidden;
            transition: box-shadow 0.2s, transform 0.2s, background 0.2s;
            animation: fadeInUp 0.7s cubic-bezier(.39,.575,.56,1) both;
        }
        .modern-stat-card:hover {
            box-shadow: 0 12px 48px 0 #6366f1aa, 0 2px 8px #38bdf855;
            background: rgba(236,245,255,0.97);
            transform: translateY(-4px) scale(1.03);
        }
        .modern-stat-card .stat-icon {
            width: 32px;
            height: 32px;
            border-radius: 10px;
            font-size: 1.1rem;
            margin-bottom: 0.4rem;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 2px 8px #6366f122;
            background: linear-gradient(135deg, #6366f1 60%, #38bdf8 100%);
            color: #fff;
            transition: box-shadow 0.2s, background 0.2s;
        }
        .modern-stat-card.border-primary { border-left-color: #6366f1; }
        .modern-stat-card.border-success { border-left-color: #22c55e; }
        .modern-stat-card.border-info { border-left-color: #0ea5e9; }
        .modern-stat-card.border-warning { border-left-color: #f59e42; }
        .modern-stat-card.border-danger { border-left-color: #ef4444; }
        .modern-stat-card .stat-label {
            color: #6e7891;
            font-size: 0.75rem;
            font-weight: 600;
            margin-bottom: 0.3rem;
            line-height: 1.2;
        }
        .modern-stat-card .stat-value {
            color: #1e293b;
            font-size: 1.1rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            line-height: 1.2;
        }
        .modern-stat-card .stat-progress-outer {
            height: 6px;
            margin-top: auto;
            width: 100%;
            background: #e9ecef;
            border-radius: 4px;
            overflow: hidden;
        }
        .modern-stat-card .stat-progress {
            height: 100%;
            border-radius: 4px;
            transition: width 0.5s;
        }
        .modern-stat-card .bg-primary { background: linear-gradient(90deg,#6366f1 60%,#818cf8 100%)!important; }
        .modern-stat-card .bg-success { background: linear-gradient(90deg,#22c55e 60%,#4ade80 100%)!important; }
        .modern-stat-card .bg-info { background: linear-gradient(90deg,#0ea5e9 60%,#38bdf8 100%)!important; }
        .modern-stat-card .bg-warning { background: linear-gradient(90deg,#f59e42 60%,#fbbf24 100%)!important; }
        @keyframes fadeInUp {
            0% { opacity: 0; transform: translateY(40px); }
            100% { opacity: 1; transform: none; }
        }
        .card {
            border-radius: 22px;
            box-shadow: 0 8px 32px rgba(37,99,235,0.13), 0 1.5px 6px rgba(0,0,0,0.03);
            border: none;
            background: rgba(255,255,255,0.85);
            backdrop-filter: blur(6px);
            transition: box-shadow 0.2s, transform 0.2s;
        }
        .card:hover {
            box-shadow: 0 12px 48px 0 #6366f1aa, 0 2px 8px #38bdf855;
            transform: translateY(-4px) scale(1.01);
        }
        .card-header {
            background: none;
            border-bottom: none;
            padding: 1.2rem 1.5rem 0.7rem 1.5rem;
        }
        .card-title {
            color: #6366f1;
            font-weight: 700;
            font-size: 1.2rem;
            letter-spacing: 0.01em;
        }
        .chart-container {
            position: relative;
            height: 320px;
            width: 100%;
            background: rgba(255,255,255,0.7);
            border-radius: 18px;
            box-shadow: 0 2px 16px #6366f122;
            padding: 1.2rem 1rem 1rem 1rem;
            margin-bottom: 1.5rem;
        }
        .table {
            background: rgba(255,255,255,0.95);
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 2px 12px #6366f122;
            margin-bottom: 1.5rem;
            animation: fadeInUp 0.7s cubic-bezier(.39,.575,.56,1) both;
        }
        .table th, .table td {
            vertical-align: middle;
            font-size: 1rem;
            padding: 1rem;
        }
        .table th {
            background: #f3f4f6;
            color: #6366f1;
            font-weight: 700;
            border-bottom: 2px solid #e0e7ff;
        }
        .table td {
            color: #232d3b;
            border-bottom: 1px solid #e0e7ff;
        }
        .badge {
            border-radius: 8px;
            font-size: 0.92rem;
            font-weight: 600;
            padding: 0.5em 1.1em;
            box-shadow: 0 1px 4px #6366f122;
            margin-right: 0.5em;
        }
        .bg-success { background: linear-gradient(90deg,#22c55e 60%,#4ade80 100%)!important; color: #fff!important; }
        .bg-danger { background: linear-gradient(90deg,#ef4444 60%,#f87171 100%)!important; color: #fff!important; }
        .bg-warning { background: linear-gradient(90deg,#f59e42 60%,#fbbf24 100%)!important; color: #fff!important; }
        .bg-info { background: linear-gradient(90deg,#0ea5e9 60%,#38bdf8 100%)!important; color: #fff!important; }
        .bg-primary { background: linear-gradient(90deg,#6366f1 60%,#818cf8 100%)!important; color: #fff!important; }
        .bg-secondary { background: linear-gradient(90deg,#64748b 60%,#a1a1aa 100%)!important; color: #fff!important; }
        .fade-in {
            animation: fadeInUp 0.7s cubic-bezier(.39,.575,.56,1) both;
        }
        @media (max-width: 768px) {
            .modern-stat-card, .card, .chart-container, .table {
                border-radius: 14px;
                padding: 0.7rem 0.7rem 0.7rem 0.7rem;
            }
            .modern-stat-card .stat-icon { width: 28px; height: 28px; font-size: 1rem; }
            .modern-stat-card .stat-value { font-size: 1.3rem; }
            .card-header { padding: 0.7rem 1rem 0.5rem 1rem; }
            .chart-container { height: 200px; }
        }

        /* Permission Modal Blur Effect */
        .modal-blur .wrapper {
            filter: blur(8px);
            transition: filter 0.3s ease;
        }

        .modal-blur .main-content {
            filter: blur(8px);
            transition: filter 0.3s ease;
        }

        .modal-blur .container-fluid {
            filter: blur(8px);
            transition: filter 0.3s ease;
        }

        /* Ensure modal and its backdrop are not blurred */
        .modal {
            filter: none !important;
        }

        .modal-backdrop {
            filter: none !important;
        }

        .modal-dialog {
            filter: none !important;
        }

        .modal-content {
            filter: none !important;
        }

        /* Permission modal specific styling */
        #permissionModal .modal-content {
            border: none;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            backdrop-filter: blur(10px);
            background: rgba(255, 255, 255, 0.95);
        }

        #permissionModal .modal-header {
            border-bottom: 1px solid rgba(0, 0, 0, 0.1);
            border-radius: 20px 20px 0 0;
        }

        #permissionModal .modal-footer {
            border-top: 1px solid rgba(0, 0, 0, 0.1);
            border-radius: 0 0 20px 20px;
        }

        /* Override Bootstrap modal backdrop for better blur effect */
        .modal-backdrop.show {
            opacity: 0.3;
        }
    </style>
</head>
<body>
<div class="wrapper">
        <?php include '../includes/sidebar.php'; ?>
    <div class="main-content">
            <?php include '../includes/topbar.php'; ?>
            
        <div class="container-fluid py-4">
            <!-- Date Filter -->
            <div class="date-filter mb-5">
                <form method="GET" class="row g-3 align-items-center">
                    <div class="col-md-3 col-sm-6">
                        <label class="form-label">Start Date</label>
                        <input type="date" name="start_date" class="form-control" value="<?php echo $start_date; ?>">
                    </div>
                    <div class="col-md-3 col-sm-6">
                        <label class="form-label">End Date</label>
                        <input type="date" name="end_date" class="form-control" value="<?php echo $end_date; ?>">
                    </div>
                    <div class="col-md-3 col-sm-6">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary d-block w-100">Apply Filter</button>
                    </div>
                </form>
            </div>

            <!-- Stat Cards Row -->
            <div class="row g-2 mb-4 mt-4">
                <div class="col-xl col-lg col-md-4 col-sm-6">
                    <div class="modern-stat-card border-primary">
                        <div class="stat-icon bg-primary text-white"><i class="fas fa-users"></i></div>
                        <div class="stat-label">Total Customers</div>
                        <div class="stat-value"><?php echo number_format($total_customers); ?></div>
                        <div class="stat-progress-outer">
                            <div class="stat-progress bg-primary" style="width: 80%"></div>
                        </div>
                    </div>
                </div>
                <div class="col-xl col-lg col-md-4 col-sm-6">
                    <div class="modern-stat-card border-success">
                        <div class="stat-icon bg-success text-white"><i class="fas fa-shopping-cart"></i></div>
                        <div class="stat-label">Today's Cash Sales</div>
                        <div class="stat-value">Ksh <?php echo number_format($today_cash_sales_amount, 2); ?></div>
                        <div class="stat-progress-outer">
                            <div class="stat-progress bg-success" style="width: 60%"></div>
                        </div>
                    </div>
                </div>
                <div class="col-xl col-lg col-md-4 col-sm-6">
                    <div class="modern-stat-card border-info">
                        <div class="stat-icon bg-info text-white"><i class="fas fa-truck"></i></div>
                        <div class="stat-label">Total Suppliers</div>
                        <div class="stat-value"><?php echo number_format($total_suppliers); ?></div>
                        <div class="stat-progress-outer">
                            <div class="stat-progress bg-info" style="width: 90%"></div>
                        </div>
                    </div>
                </div>
                <div class="col-xl col-lg col-md-4 col-sm-6">
                    <div class="modern-stat-card border-warning">
                        <div class="stat-icon bg-warning text-white"><i class="fas fa-credit-card"></i></div>
                        <div class="stat-label">Today's Credit Balance</div>
                        <div class="stat-value">Ksh <?php echo number_format($today_credit_sales, 2); ?></div>
                        <div class="stat-progress-outer">
                            <div class="stat-progress bg-warning" style="width: 30%"></div>
                        </div>
                    </div>
                </div>
                <div class="col-xl col-lg col-md-4 col-sm-6">
                    <div class="modern-stat-card border-danger">
                        <div class="stat-icon bg-danger text-white"><i class="fas fa-money-bill-wave"></i></div>
                        <div class="stat-label">Total Credit Amount</div>
                        <div class="stat-value">Ksh <?php echo number_format($total_credit_amount, 2); ?></div>
                        <div class="stat-progress-outer">
                            <div class="stat-progress bg-danger" style="width: 60%"></div>
                        </div>
                    </div>
                </div>
            </div>
            <!-- End Stat Cards Row -->

            <!-- Main Content Row -->
            <div class="row g-4">
                <!-- Left Column: Sales Trend & Recent Transactions -->
                <div class="col-xl-8 col-lg-7">
                    <!-- Sales Chart -->
                    <div class="card mb-4 shadow-lg border-0" style="background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); border-radius: 1rem;">
                        <div class="card-header d-flex justify-content-between align-items-center flex-wrap bg-transparent border-0" style="border-radius: 1rem 1rem 0 0;">
                            <h5 class="card-title mb-0">Sales Trend</h5>
                            <div>
                                <select id="salesTrendRange" class="form-select form-select-sm" style="min-width: 120px;">
                                    <option value="7">Last 7 Days</option>
                                    <option value="30" selected>Last 30 Days</option>
                                    <option value="90">Last 90 Days</option>
                                </select>
                            </div>
                        </div>
                        <div class="card-body" style="background: transparent;">
                            <div class="chart-container position-relative" style="min-height: 320px;">
                                <canvas id="salesChart"></canvas>
                                <div id="salesChartLoading" class="position-absolute top-50 start-50 translate-middle" style="display: none;">
                                    <div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- Recent Transactions -->
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
                            <h5 class="card-title mb-0">Recent Transactions</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Invoice</th>
                                            <th>Customer</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_transactions as $sale): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($sale['invoice_number']); ?></td>
                                            <td><?php echo htmlspecialchars($sale['customer_name'] ?? 'Walk-in Customer'); ?></td>
                                            <td><?php echo number_format($sale['total_amount'], 2); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $sale['status'] === 'completed' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($sale['status']); ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    <!-- Low Stock Alert -->
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
                            <h5 class="card-title mb-0">Low Stock Alert</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Status</th>
                                            <th>Current Stock</th>
                                            <th>Alert Level</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($low_stock as $product): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($product['name']); ?></td>
                                            <td>
                                                <?php
                                                $status = '';
                                                $qty = $product['quantity'];
                                                $alert = $product['alert_quantity'];
                                                if ($qty <= 0) {
                                                    $status = '<span class="badge bg-danger">Out of Stock</span>';
                                                } elseif ($qty <= $alert) {
                                                    $status = '<span class="badge bg-warning">Low</span>';
                                                } else {
                                                    $status = '<span class="badge bg-success">In Stock</span>';
                                                }
                                                echo $status;
                                                ?>
                                            </td>
                                            <td><?php echo number_format($product['quantity']); ?></td>
                                            <td><?php echo number_format($product['alert_quantity']); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                <!-- Right Column: Best/Worst Selling, Top Products, Pending Credits -->
                <div class="col-xl-4 col-lg-5">
                    <!-- Best Selling Products Chart -->
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
                            <h5 class="card-title mb-0">Best Selling Products</h5>
                        </div>
                        <div class="card-body">
                            <canvas id="bestProductsChart" height="120"></canvas>
                        </div>
                    </div>
                    <!-- Worst Selling Products Chart -->
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
                            <h5 class="card-title mb-0">Worst Selling Products</h5>
                        </div>
                        <div class="card-body">
                            <canvas id="worstProductsChart" height="120"></canvas>
                        </div>
                    </div>
                    <!-- Top Selling Products Table -->
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
                            <h5 class="card-title mb-0">Top Selling Products</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Qty</th>
                                            <th>Revenue</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($top_products as $product): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($product['name']); ?></td>
                                            <td><?php echo number_format($product['total_quantity']); ?></td>
                                            <td><?php echo number_format($product['total_revenue'], 2); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    <!-- Pending Credit Sales Table -->
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
                            <h5 class="card-title mb-0">Pending Credit Sales</h5>
                            <a href="credit.php" class="btn btn-sm btn-primary mt-2 mt-sm-0">View All</a>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Customer</th>
                                            <th>Amount</th>
                                            <th>Paid</th>
                                            <th>Balance</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($pending_credits as $credit): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($credit['customer_name']); ?></td>
                                            <td><?php echo number_format($credit['credit_amount'], 2); ?></td>
                                            <td><?php echo number_format($credit['amount_paid'], 2); ?></td>
                                            <td><?php echo number_format($credit['credit_amount'] - $credit['amount_paid'], 2); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <!-- End Main Content Row -->
        </div>
    </div>
</div>

<!-- End of Dashboard Content -->

<!-- Permission Access Modal -->
<?php if (!$hasDashboardPermission): ?>
<div class="modal fade" id="permissionModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Access Restricted
                </h5>
            </div>
            <div class="modal-body text-center">
                <div class="mb-4">
                    <i class="fas fa-user-lock fa-3x text-warning mb-3"></i>
                    <h4 class="text-dark">Hello, <?php echo htmlspecialchars($userName); ?>!</h4>
                    <p class="text-muted">You don't have permission to access the Dashboard.</p>
                    <p class="text-info"><strong>Please contact Super Admin for Permission</strong></p>
                </div>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Redirecting you to a page you have access to...</strong>
                </div>
            </div>
            <div class="modal-footer justify-content-center">
                <button type="button" class="btn btn-primary" onclick="redirectToAccessiblePage()">
                    <i class="fas fa-arrow-right me-2"></i>
                    Continue
                </button>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Low Stock Modal -->
<div class="modal fade" id="lowStockModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">All Low Stock Products</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Product</th>
                                <th>Status</th>
                                <th>Current Stock</th>
                                <th>Alert Level</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            // Fetch all products for the modal
                            $stmt = $pdo->prepare("SELECT name, quantity, alert_quantity FROM products ORDER BY quantity ASC");
                            $stmt->execute();
                            $all_products = $stmt->fetchAll();
                            foreach ($all_products as $product):
                            ?>
                            <tr>
                                <td><?php echo htmlspecialchars($product['name']); ?></td>
                                <td>
                                    <?php
                                    $status = '';
                                    $qty = $product['quantity'];
                                    $alert = $product['alert_quantity'];
                                    if ($qty <= 0) {
                                        $status = '<span class="badge bg-danger">Out of Stock</span>';
                                    } elseif ($qty <= $alert) {
                                        $status = '<span class="badge bg-warning">Low</span>';
                                    } else {
                                        $status = '<span class="badge bg-success">In Stock</span>';
                                    }
                                    echo $status;
                                    ?>
                                </td>
                                <td><?php echo number_format($product['quantity']); ?></td>
                                <td><?php echo number_format($product['alert_quantity']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Refund Modal -->
<div class="modal fade" id="refundModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Process Return</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="refundForm">
                    <input type="hidden" id="refundSaleId" name="sale_id">
                    <div class="table-responsive">
                        <table class="table" id="refundItemsTable">
                            <thead>
                                <tr>
                                    <th>Product</th>
                                    <th>Quantity</th>
                                    <th>Price</th>
                                    <th>Return Qty</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Will be populated by JavaScript -->
                            </tbody>
                        </table>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Return Reason</label>
                        <select class="form-control" name="refund_reason" id="refundReason" required onchange="handleRefundReasonChange()">
                            <option value="">Select a reason</option>
                            <option value="good">Good Condition</option>
                            <option value="damaged">Damaged</option>
                            <option value="opened">Opened</option>
                            <option value="others">Others</option>
                        </select>
                    </div>
                    <div class="mb-3" id="otherReasonDiv" style="display: none;">
                        <label class="form-label">Specify Other Reason</label>
                        <textarea class="form-control" name="other_reason" id="otherReason"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Return Amount</label>
                        <input type="number" class="form-control" id="refundAmount" name="refund_amount" readonly>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="processRefund()">Process Return</button>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Permission Modal Functions
<?php if (!$hasDashboardPermission): ?>
// Multiple ways to ensure modal opens immediately
function showPermissionModal() {
    console.log('Attempting to show permission modal...');
    const modalElement = document.getElementById('permissionModal');
    
    if (modalElement) {
        console.log('Modal element found, showing modal...');
        const permissionModal = new bootstrap.Modal(modalElement);
        
        // Add blur effect to background elements
        const wrapper = document.querySelector('.wrapper');
        const mainContent = document.querySelector('.main-content');
        const container = document.querySelector('.container-fluid');
        
        if (wrapper) wrapper.classList.add('modal-blur');
        if (mainContent) mainContent.classList.add('modal-blur');
        if (container) container.classList.add('modal-blur');
        
        console.log('Blur classes added to background elements');
        
        permissionModal.show();
        console.log('Permission modal shown successfully');
        
        // Remove blur when modal is hidden
        modalElement.addEventListener('hidden.bs.modal', function() {
            console.log('Modal hidden, removing blur...');
            if (wrapper) wrapper.classList.remove('modal-blur');
            if (mainContent) mainContent.classList.remove('modal-blur');
            if (container) container.classList.remove('modal-blur');
            console.log('Blur classes removed');
        });
    } else {
        console.error('Modal element not found!');
    }
}

// Try multiple event listeners to ensure modal shows
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOMContentLoaded - Permission modal should show for user: <?php echo htmlspecialchars($userName); ?>');
    
    // Apply blur immediately
    const wrapper = document.querySelector('.wrapper');
    const mainContent = document.querySelector('.main-content');
    const container = document.querySelector('.container-fluid');
    
    if (wrapper) wrapper.classList.add('modal-blur');
    if (mainContent) mainContent.classList.add('modal-blur');
    if (container) container.classList.add('modal-blur');
    
    console.log('Blur applied immediately on page load');
    
    showPermissionModal();
});

// Also try on window load as backup
window.addEventListener('load', function() {
    console.log('Window load - Checking if modal needs to be shown...');
    const modalElement = document.getElementById('permissionModal');
    if (modalElement && !modalElement.classList.contains('show')) {
        console.log('Modal not shown yet, showing now...');
        
        // Ensure blur is applied
        const wrapper = document.querySelector('.wrapper');
        const mainContent = document.querySelector('.main-content');
        const container = document.querySelector('.container-fluid');
        
        if (wrapper) wrapper.classList.add('modal-blur');
        if (mainContent) mainContent.classList.add('modal-blur');
        if (container) container.classList.add('modal-blur');
        
        showPermissionModal();
    }
});

function redirectToAccessiblePage() {
    console.log('Redirecting user to accessible page...');
    // Determine the best page to redirect to based on user permissions
    <?php
    $redirectUrl = '';
    if (hasPermission('view_sales') || hasPermission('create_sales')) {
        $redirectUrl = 'pos.php';
    } elseif (hasPermission('view_products') || hasPermission('manage_products')) {
        $redirectUrl = 'products.php';
    } elseif (hasPermission('view_customers') || hasPermission('manage_customers')) {
        $redirectUrl = 'customers.php';
    } elseif (hasPermission('view_suppliers') || hasPermission('manage_suppliers')) {
        $redirectUrl = 'suppliers.php';
    } elseif (hasPermission('view_inventory') || hasPermission('manage_inventory')) {
        $redirectUrl = 'inventory.php';
    } elseif (hasPermission('view_reports')) {
        $redirectUrl = 'reports.php';
    } else {
        $redirectUrl = '../auth/login.php';
    }
    ?>
    console.log('Redirecting to: <?php echo $redirectUrl; ?>');
    window.location.href = '<?php echo $redirectUrl; ?>';
}
<?php endif; ?>

// Sales Chart
const ctx = document.getElementById('salesChart').getContext('2d');
const salesData = <?php echo json_encode($daily_sales); ?>;
const salesChart = new Chart(ctx, {
    type: 'line',
    data: {
        labels: salesData.map(item => item.date),
        datasets: [
            {
                label: 'Total Sales',
                data: salesData.map(item => item.total),
                borderColor: 'rgba(56,116,255,1)',
                backgroundColor: 'rgba(56,116,255,0.08)',
                tension: 0.45,
                fill: true,
                pointRadius: 3,
                pointHoverRadius: 6,
                pointStyle: 'circle',
                borderWidth: 3,
            },
            {
                label: 'Credit Sales',
                data: salesData.map(item => item.credit_total),
                borderColor: 'rgba(0,210,122,1)',
                backgroundColor: 'rgba(0,210,122,0.08)',
                tension: 0.45,
                fill: true,
                pointRadius: 3,
                pointHoverRadius: 6,
                pointStyle: 'circle',
                borderWidth: 3,
            }
        ]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'top',
                labels: {
                    usePointStyle: true,
                    font: { size: 15, family: 'Inter, Arial, sans-serif' },
                    color: '#1e293b',
                },
            },
            tooltip: {
                mode: 'index',
                intersect: false,
                backgroundColor: '#fff',
                titleColor: '#1e293b',
                bodyColor: '#334155',
                borderColor: '#3874ff',
                borderWidth: 1,
                padding: 12,
                caretSize: 8,
                cornerRadius: 8,
                displayColors: true,
                callbacks: {
                    label: function(context) {
                        return `${context.dataset.label}: ${context.parsed.y.toLocaleString()}`;
                    }
                }
            }
        },
        interaction: {
            mode: 'nearest',
            axis: 'x',
            intersect: false
        },
        scales: {
            x: {
                grid: { display: false },
                ticks: { color: '#64748b', font: { size: 13 } }
            },
            y: {
                beginAtZero: true,
                grid: { color: 'rgba(100,116,139,0.08)' },
                ticks: { color: '#64748b', font: { size: 13 } }
            }
        },
        animation: {
            duration: 1200,
            easing: 'easeOutQuart'
        }
    }
});

// Best Selling Products Chart
const bestLabels = <?php echo json_encode(array_column($top_products, 'name')); ?>;
const bestData = <?php echo json_encode(array_map('intval', array_column($top_products, 'total_quantity'))); ?>;
const bestCtx = document.getElementById('bestProductsChart').getContext('2d');
new Chart(bestCtx, {
    type: 'bar',
    data: {
        labels: bestLabels,
        datasets: [{
            label: 'Quantity Sold',
            data: bestData,
            backgroundColor: 'rgba(34,197,94,0.7)'
        }]
    },
    options: {
        responsive: true,
        plugins: { legend: { display: false } },
        scales: { y: { beginAtZero: true } }
    }
});
// Worst Selling Products Chart
const worstLabels = <?php echo json_encode(array_column($worst_products, 'name')); ?>;
const worstData = <?php echo json_encode(array_map('intval', array_column($worst_products, 'total_quantity'))); ?>;
const worstCtx = document.getElementById('worstProductsChart').getContext('2d');
new Chart(worstCtx, {
    type: 'bar',
    data: {
        labels: worstLabels,
        datasets: [{
            label: 'Quantity Sold',
            data: worstData,
            backgroundColor: 'rgba(239,68,68,0.7)'
        }]
    },
    options: {
        responsive: true,
        plugins: { legend: { display: false } },
        scales: { y: { beginAtZero: true } }
    }
});

function handleRefundReasonChange() {
    const reason = $('#refundReason').val();
    const otherReasonDiv = $('#otherReasonDiv');
    const refundAmount = $('#refundAmount');
    
    if (reason === 'others') {
        otherReasonDiv.show();
        $('#otherReason').prop('required', true);
    } else {
        otherReasonDiv.hide();
        $('#otherReason').prop('required', false);
    }
    
    // Only auto-calculate for 'good' condition
    if (reason === 'good') {
        refundAmount.prop('readonly', true);
        updateRefundAmount();
    } else {
        // Allow manual input for other conditions
        refundAmount.prop('readonly', false);
        // Don't auto-calculate for other conditions
    }
}

function updateRefundAmount() {
    const reason = $('#refundReason').val();
    if (reason === 'good') {
        let total = 0;
        $('.return-qty').each(function() {
            const qty = parseInt($(this).val()) || 0;
            const price = parseFloat($(this).data('price'));
            total += qty * price;
        });
        $('#refundAmount').val(total.toFixed(2));
    }
}

// Add event listener for manual refund amount changes
$('#refundAmount').on('input', function() {
    const reason = $('#refundReason').val();
    if (reason !== 'good') {
        // Validate the input
        let value = parseFloat($(this).val()) || 0;
        if (value < 0) {
            value = 0;
        }
        $(this).val(value.toFixed(2));
    }
});

function processRefund() {
    const saleId = $('#refundSaleId').val();
    const refundAmount = $('#refundAmount').val();
    const refundReason = $('#refundReason').val();
    const otherReason = $('#otherReason').val();
    
    if (!refundReason) {
        alert('Please select a return reason');
        return;
    }
    
    if (refundReason === 'others' && !otherReason) {
        alert('Please specify the other reason');
        return;
    }
    
    const refundItems = [];
    $('.return-qty').each(function() {
        const qty = parseInt($(this).val()) || 0;
        if (qty > 0) {
            refundItems.push({
                product_id: $(this).data('product-id'),
                quantity: qty,
                price: $(this).data('price')
            });
        }
    });
    
    if (refundItems.length === 0) {
        alert('Please select items to return');
        return;
    }

    const refundData = {
        sale_id: saleId,
        refund_amount: refundAmount,
        refund_reason: refundReason === 'others' ? otherReason : refundReason,
        refund_items: refundItems
    };
    
    // Show loading state
    const submitButton = $('#refundModal .btn-primary');
    const originalText = submitButton.html();
    submitButton.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Processing...');
    
    // Log the request data
    console.log('Sending refund request:', refundData);
    
    $.ajax({
        url: '/POS/api/process_refund.php',  // Using absolute path
        method: 'POST',
        contentType: 'application/json',
        data: JSON.stringify(refundData),
        dataType: 'json',
        success: function(response) {
            console.log('Refund response:', response);
            if (response.success) {
                alert('Return processed successfully');
                $('#refundModal').modal('hide');
                // Print refund receipt
                window.open('/POS/receipts/print_refund_receipt.php?refund_id=' + response.refund_id, '_blank');
                // Refresh the transactions table
                $('#allTransactionsTable').DataTable().ajax.reload();
            } else {
                alert(response.message || 'Error processing refund');
            }
        },
        error: function(xhr, status, error) {
            console.error('Refund Error Details:');
            console.error('Status:', status);
            console.error('Error:', error);
            console.error('Response Text:', xhr.responseText);
            console.error('Response Status:', xhr.status);
            console.error('Response Headers:', xhr.getAllResponseHeaders());
            
            let errorMessage = 'Error processing refund. Please try again.';
            try {
                const response = JSON.parse(xhr.responseText);
                if (response.message) {
                    errorMessage = response.message;
                }
            } catch (e) {
                console.error('Error parsing response:', e);
                // Check if the response contains HTML
                if (xhr.responseText.includes('<!DOCTYPE')) {
                    errorMessage = 'Server error occurred. Please check the server logs.';
                    // Log the full HTML response for debugging
                    console.error('HTML Response:', xhr.responseText);
                }
            }
            alert(errorMessage);
        },
        complete: function() {
            // Reset button state
            submitButton.prop('disabled', false).html(originalText);
        }
    });
}

function showRefundModal(saleId) {
    $('#refundSaleId').val(saleId);
    $('#refundItemsTable tbody').empty();
    $('#refundAmount').val('0');
    $('#refundReason').val('');
    $('#otherReason').val('');
    $('#otherReasonDiv').hide();
    
    // Show loading state
    const modalBody = $('#refundModal .modal-body');
    const originalContent = modalBody.html();
    modalBody.html('<div class="text-center"><i class="fas fa-spinner fa-spin fa-2x"></i><br>Loading sale items...</div>');
    
    // Fetch sale items
    $.ajax({
        url: '/POS/api/get_sale_items.php',
        method: 'GET',
        data: { sale_id: saleId },
        dataType: 'json',
        success: function(response) {
            console.log('Sale items response:', response);
            if (response.success) {
                modalBody.html(originalContent);
                response.items.forEach(function(item) {
                    const row = `
                        <tr>
                            <td>${item.name}</td>
                            <td>${item.quantity}</td>
                            <td>${parseFloat(item.unit_price).toFixed(2)}</td>
                            <td>
                                <input type="number" class="form-control return-qty" 
                                    max="${item.quantity}" min="0" value="0"
                                    data-price="${item.unit_price}"
                                    data-product-id="${item.product_id}"
                                    onchange="updateRefundAmount()">
                            </td>
                        </tr>
                    `;
                    $('#refundItemsTable tbody').append(row);
                });
                $('#refundModal').modal('show');
            } else {
                modalBody.html(originalContent);
                alert(response.message || 'Error loading sale items');
            }
        },
        error: function(xhr, status, error) {
            console.error('Error loading sale items:', {
                status: status,
                error: error,
                response: xhr.responseText
            });
            modalBody.html(originalContent);
            alert('Error loading sale items. Please try again.');
        }
    });
}
</script>
<?php include '../includes/footer.php'; ?>
</body>
</html> 